/*:
 * @target MZ
 * @plugindesc Backlog Close Click Debounce — バックログを閉じた直後のクリックを数フレーム無効化して誤反応を防止
 * @author HS
 *
 * @param debounceFrames
 * @text 無効化フレーム数
 * @type number
 * @min 1
 * @max 30
 * @default 6
 * @desc バックログを閉じてからこのフレーム数の間、ピクチャボタンのクリックを無視します（60fpsで6=約100ms）。
 */
(() => {
  'use strict';
  const PN = document.currentScript.src.split('/').pop().replace(/\.js$/,'');
  const P  = PluginManager.parameters(PN);
  const DEBOUNCE = Math.max(1, +P.debounceFrames || 6);

  let prevOpen = false;
  let suppressUntil = 0;

  // HSBL（Backlog Core）が expose しているフラグを監視
  const isBacklogOpen = () => !!(window.HSBL && typeof HSBL.isOpen === 'function' && HSBL.isOpen());

  // Sceneごとに監視して、閉じた瞬間からDEBOUNCEフレーム食う
  const _SceneBase_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _SceneBase_update.apply(this, arguments);
    const nowOpen = isBacklogOpen();
    if (prevOpen && !nowOpen) {
      suppressUntil = Graphics.frameCount + DEBOUNCE;
      // 念のため入力バッファも掃除
      try { Input.clear?.(); TouchInput.clear?.(); } catch(_) {}
    }
    prevOpen = nowOpen;
  };

  // 他プラグインからも参照できるように
  window.HS = window.HS || {};
  HS.isUiClickSuppressed = () => Graphics.frameCount < suppressUntil;

  // ButtonPicture系のクリック入口をガード
  const _isClickEnabled = Sprite_Picture.prototype.isClickEnabled;
  Sprite_Picture.prototype.isClickEnabled = function() {
    if (HS.isUiClickSuppressed?.()) return false;
    return _isClickEnabled ? _isClickEnabled.apply(this, arguments) : true;
  };

  const _onClick = Sprite_Picture.prototype.onClick;
  Sprite_Picture.prototype.onClick = function() {
    if (HS.isUiClickSuppressed?.()) return; // ここで最終ガード
    if (_onClick) _onClick.apply(this, arguments);
  };
})();
